/*
Copyright (C) Lingodigit Co., Ltd. - All Rights Reserved
Author: Hung-Ming Chen <hmchen@lingodigit.com>
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <getopt.h>
#include <signal.h>
#include <inttypes.h>
#include <time.h>

#include <gst/gst.h>
#include <gst/rtsp/gstrtsp.h>
#include <glib/gprintf.h>

enum AudioType {
    AUDIO_NOT_SUPPORT=0,
    AUDIO_MPEG4_GENERIC,
    AUDIO_MP4A_LATM
};

struct Camera {
    gchar *name;
    gchar *url;
    gboolean auth;
    gchar *user;
    gchar *pass;
    // internal settings
    GstElement *pipeline;
    int has_video;
    int has_audio;
    guint ev_source;
    gchar outdir[256];
};

// generic
char *basedir = "/tmp/hls";
int64_t udp_timeout=20000000; // 20sec
int64_t tcp_timeout=20000000; // 20sec
int target_duration = 1;
int max_files = 5;
int force_tcp = 0;
int latency=50; // network caching latency, millisecond

gchar *conf_file = "/etc/hls.conf";
GList *camera_list = NULL;
GMainLoop *loop;
int gst_quit;

int camera_connect(struct Camera *camera);
gboolean camera_reconnect(gpointer data);
void camera_set_state(struct Camera *camera, char *state);
void sigint_handle(int num);

static gboolean
bus_handler (GstBus * bus, GstMessage * message, gpointer data)
{
    GError *err = NULL;
    gchar *debug = NULL;
    int retry = 0;

    struct Camera *camera = (struct Camera *)data;
    switch (message->type) {
    case GST_MESSAGE_EOS:
        g_debug("%s: EOF", __func__);
        if(!gst_quit)
            retry = 1;
        break;
    case GST_MESSAGE_WARNING:
        gst_message_parse_warning (message, &err, &debug);
        g_debug("%s", debug);
        g_message("[%s] %s", g_quark_to_string(err->domain), err->message);
        g_error_free(err);
        g_free(debug);
        break;
    case GST_MESSAGE_ERROR: {
        gst_message_parse_error (message, &err, &debug);
        g_debug("%s", debug);
        g_message("[%s] %s", g_quark_to_string(err->domain), err->message);
        g_error_free(err);
        g_free(debug);

        if(camera->pipeline) { // prevent duplicate timeout source
            retry = 1;
        }
        break;
    }
    default:
        break;
    }

    if(retry && camera->pipeline) { // prevent duplicate timeout source
        g_timeout_add_seconds(5, camera_reconnect, data);
        gst_element_set_state(camera->pipeline, GST_STATE_NULL);
        gst_object_unref(camera->pipeline);
        camera->pipeline = NULL;
        camera_set_state(camera, "DICONNECTED");
    }

    return TRUE;
}

gboolean remove_temp_rtspsrc(gpointer user_data)
{
    //g_debug("%s", __func__);
    GstElement *src = (GstElement *)user_data;
    gst_element_set_state(src, GST_STATE_NULL);
    gst_object_unref(src);
    return FALSE;
}

gboolean stop_temp_rtspsrc(gpointer user_data)
{
    //g_debug("%s", __func__);
    GstElement *src = (GstElement *)user_data;
    gst_element_set_state(src, GST_STATE_READY);
    g_timeout_add(50, remove_temp_rtspsrc, user_data); // 50ms delay which allow rtspsrc to send TEARDOWN
    return FALSE;
}

gboolean create_hls_pipeline(gpointer user_data)
{
    struct Camera *camera = (struct Camera *)user_data;
    GError *gerr = NULL;
    gchar launch[1024];
    GstBus *bus;
    gchar *protocols="";

    if(force_tcp)
        protocols = "protocols=tcp";

    // common part (h.264 video)
    g_sprintf(launch,
              "rtspsrc name=src timeout=%"PRId64 " tcp-timeout=%"PRId64 " latency=%d location=%s %s ! queue ! rtph264depay ! h264parse config-interval=1 ! "
              "video/x-h264,stream-format=byte-stream,alignment=au ! "
              "mpegtsmux name=mux ! hlssink target-duration=%d max-files=%d playlist-location=%s/playlist.m3u8 location=%s/segment%%05d.ts",
              udp_timeout, tcp_timeout, latency, camera->url, protocols,
              target_duration, max_files, camera->outdir, camera->outdir);

    if(camera->has_audio == AUDIO_MPEG4_GENERIC)
        g_strlcat(launch, " src. ! rtpmp4gdepay ! aacparse ! mux.", 1024);
    else if(camera->has_audio == AUDIO_MP4A_LATM)
        g_strlcat(launch, " src. ! rtpmp4adepay ! mux.", 1024);

    camera->pipeline = gst_parse_launch(launch, &gerr);

    if(!camera->pipeline) {
        g_print("parse launch failed: %s\n", launch);
        g_print("error message: %s\n", gerr->message);
        camera_set_state(camera, "ERROR:PIPELINE");
        goto quit;
    }

    // rtsp authentication
    if(camera->auth) {
        GstElement *src = gst_bin_get_by_name(GST_BIN(camera->pipeline), "src");
        g_object_set(G_OBJECT(src), "user-id", camera->user, "user-pw", camera->pass, NULL);
    }

    // watch bus error message for disconnect detection
    bus = gst_element_get_bus (camera->pipeline);
    gst_bus_add_watch (bus, bus_handler, camera);
    gst_object_unref (bus);

    gst_element_set_state(camera->pipeline, GST_STATE_PLAYING);
    camera_set_state(camera, "CONNECTED");
    g_debug("session start: %s", camera->url);

quit:
    //g_free(launch);
    return FALSE;
}

void rtspsrc_on_sdp(GstElement* object,
                    GstSDPMessage* arg0,
                    gpointer user_data)
{
    int i;
    int len;
    struct Camera *camera = (struct Camera *)user_data;
    const GstSDPMedia *media;
    const gchar *rtpmap;
    len = gst_sdp_message_medias_len(arg0);
    for(i=0;i<len;i++) {
        media = gst_sdp_message_get_media(arg0, i);
        rtpmap = gst_sdp_media_get_attribute_val(media, "rtpmap");
        // media cannot be empty; h.264 and aac must have rtpmap attribute
        if(!media || !rtpmap)
            continue;

        if(strcmp(media->media, "video") == 0) {
            if(strstr(rtpmap, "H264"))
                camera->has_video = 1;
        }
        else if(strcmp(media->media, "audio") == 0) {
            if(strstr(rtpmap, "MPEG-LATM"))
                camera->has_audio = AUDIO_MP4A_LATM;
            else if(strstr(rtpmap, "MPEG4-GENERIC"))
                camera->has_audio = AUDIO_MPEG4_GENERIC;
        }
    }

    g_source_remove(camera->ev_source);
    camera->ev_source = 0;

    // clean up temporary rtspsrc
    g_timeout_add(0, stop_temp_rtspsrc, object);

    if(camera->has_video) {
        g_timeout_add(50, create_hls_pipeline, camera);
    }
    else {
        g_message("no h264 stream: %s; retry in 5sec", camera->url);
        g_timeout_add_seconds(5, camera_reconnect, user_data);
        camera_set_state(camera, "ERROR:NO_H264");
    }
}

void camera_set_state(struct Camera *camera, char *state)
{
    char filename[256];
    FILE *fp;
    sprintf(filename, "%s/%s/state", basedir, camera->name);
    fp = fopen(filename, "w");
    if(!fp) {
        fprintf(stderr, "%s: cannot open %s\n", __func__, filename);
    }
    else {
        fprintf(fp, "%s\n", state);
        fclose(fp);
    }
}

gboolean camera_reconnect(gpointer data)
{
    g_debug("%s", __func__);
    struct Camera *camera = (struct Camera *)data;

    camera_connect(camera);
    return FALSE;
}

gboolean rtspsrc_connect_timeout(gpointer user_data)
{
    gchar *location;
    GstElement *src = (GstElement *)user_data;
    g_object_get(G_OBJECT(src), "location", &location, NULL);
    g_debug("%s: location=%s, retry...", __func__, location);
    // reuse exist rtspsrc
    gst_element_set_state(src, GST_STATE_NULL);
    gst_element_set_state(src, GST_STATE_PAUSED);

    return TRUE;
}

// initial connection for detecting supported audio and video streams in RTSP url
int camera_connect(struct Camera *camera)
{
    // create output directory
    g_debug("%s: %s", __func__, camera->url);
    sprintf(camera->outdir, "%s/%s", basedir, camera->name);
    mkdir(camera->outdir, 0777);

    // create a temporary rtspsrc to detect audio and video streams
    GstElement *src = gst_element_factory_make("rtspsrc", NULL);
    g_object_set(G_OBJECT(src), "location", camera->url, "timeout", udp_timeout, "tcp-timeout", tcp_timeout, NULL);
    if(camera->auth)
        g_object_set(G_OBJECT(src), "user-id", camera->user, "user-pw", camera->pass, NULL);
    if(force_tcp)
        g_object_set(G_OBJECT(src), "protocols", 4, NULL);


    g_signal_connect(src, "on-sdp", G_CALLBACK(rtspsrc_on_sdp), camera);
    gst_element_set_state(src, GST_STATE_PAUSED);

    // retry when the camera is not reachable
    camera->ev_source = g_timeout_add_seconds(20, rtspsrc_connect_timeout, src);
    camera_set_state(camera, "CONNECTING");
    return 0;
}

int load_config(const char *filename)
{
    int i;
    GError *gerr = NULL;
    GKeyFile *keyfile = g_key_file_new();
    gchar **groups;
    gsize length;
    struct Camera *cam;

    if(!g_key_file_load_from_file(keyfile, filename, G_KEY_FILE_NONE, &gerr))  {
        fprintf(stderr, "%s: %s, %s\n", __func__, filename, gerr->message);
        return -1;
    }

    basedir = g_key_file_get_string(keyfile, "generic", "basedir", NULL);
    target_duration = g_key_file_get_integer(keyfile, "generic", "target-duration", NULL);
    max_files = g_key_file_get_integer(keyfile, "generic", "max-files", NULL);

    groups = g_key_file_get_groups(keyfile, &length);
    for(i=0;i<length;i++) {
        if(strcmp(groups[i], "generic")) {
            cam = (struct Camera *)g_malloc0(sizeof(struct Camera));
            cam->name = groups[i];
            cam->url = g_key_file_get_string(keyfile, groups[i], "url", &gerr);
            cam->auth = g_key_file_get_integer(keyfile, groups[i], "auth", &gerr);
            if(cam->auth) {
                cam->user = g_key_file_get_string(keyfile, groups[i], "user", &gerr);
                cam->pass = g_key_file_get_string(keyfile, groups[i], "pass", &gerr);
            }
            camera_list = g_list_append(camera_list, cam);
        }
    }

    return 0;
}

void gfunc_dump(gpointer data, gpointer user_data)
{
    struct Camera *cam = (struct Camera *)data;
    fprintf(stderr, "[%s]\n", cam->name);
    fprintf(stderr, "url: %s\n", cam->url);
    fprintf(stderr, "auth: %d\n", cam->auth);
    if(cam->auth) {
        fprintf(stderr, "user: %s\n", cam->user);
        fprintf(stderr, "pass: %s\n", cam->pass);
    }
}

void gfunc_connect(gpointer data, gpointer user_data)
{
    camera_connect((struct Camera *)data);
}

void print_help()
{
    fprintf(stderr, "usage: gst-hls [-htvd][-f filename]\n");
    fprintf(stderr, "-h help message\n");
    fprintf(stderr, "-t force rtsp over tcp\n");
    fprintf(stderr, "-v verbose\n");
    fprintf(stderr, "-d foreground (debug mode)\n");
    fprintf(stderr, "-f filename\tfilename of the config file (default: %s)\n", conf_file);
    exit(0);
}

int main(int argc, char **argv)
{
    int opt;
    int debug = 0;

    while((opt = getopt(argc, argv, "tdf:hv")) != -1) {
        switch(opt) {
        case 't':
            force_tcp = 1;
            break;
        case 'f':
            conf_file = optarg;
            break;
        case 'h':
            print_help();
            break;
        case 'v':
            setenv("G_MESSAGES_DEBUG", "all", 1);
            break;
        case 'd':
            debug=1;
            break;
        default:
            fprintf(stderr, "unknown option %c\n", opt);
        }
    }

    if(load_config(conf_file) < 0)
        exit(1);

    //g_list_foreach(camera_list, gfunc_dump, NULL);

    // check output directory
    if(!g_file_test(basedir, G_FILE_TEST_IS_DIR)) {
        fprintf(stderr, "directory %s doesn't exist\n", basedir);
        return -1;
    }

    if(debug) {
        g_debug("debug mode");
    }
    else {
        // don't get into background for debug mode
        if(daemon(1, 1) < 0) {
            fprintf(stderr, "error: %d\n", errno);
            exit(1);
        }
        fprintf(stderr, "gst-hls started. pid=%u\n", getpid());
    }

    signal(SIGINT, sigint_handle);
    gst_init(&argc, &argv);

    loop = g_main_loop_new (NULL, FALSE);

    g_list_foreach(camera_list, gfunc_connect, NULL);

    g_main_loop_run (loop);

    return 0;
}

void sigint_handle(int num)
{
    GList *l;
    struct Camera *cam;

    g_message("catched SIGINT");
    for(l=camera_list; l!=NULL; l=l->next) {
        cam = (struct Camera *)l->data;
        if(cam->pipeline) {
            g_message("stop %s", cam->url);
            gst_element_set_state(cam->pipeline, GST_STATE_READY);
        }
    }
    usleep(50000);
    g_main_loop_quit(loop);
}
